#ifndef TREE_NODE_HPP
#define TREE_NODE_HPP

#include "Body.hpp"
#include <iterator>
#include "Box.hpp"
#include <cstdlib>
#include <vector>

namespace nbody {
	class Tree;
	//class for storing node information
	class Node {
		friend class Tree;
		friend class NodeIterator;
		friend class BarnesHutTree;
	protected:
		Box bb;
		std::vector<Body> bodies;
		Node* prev{ this };
		Node* next{ this };
		Node* nextSibling{ nullptr };
		Node* prevSibling{ nullptr };
		Node* afterSubtree{ nullptr };
		bool leaf{ true };
		Tree* tree{ nullptr };
		Body representative;
	public:
		Node(Tree* _tree);
		virtual ~Node() = default;
		virtual bool isSplitable() const;
		virtual void extendBBforBodies();
		virtual void extendBBtoCube();
		virtual Box getBB() const;
		virtual void setBB(const Box& bb_);
		virtual std::vector<Body> getBodies() const;
		virtual void insertBefore(Node* node);
		virtual void insertAfter(Node* node);
		virtual void unlink();
		virtual void update();
		virtual double getL() const;
		virtual bool isCorrect() const;
		virtual void print(std::size_t parallelId) const;
		virtual bool sufficientForBody(const Body& body) const;
		virtual bool sufficientForBox(const Box& box) const;
		virtual void setBodies(const std::vector<Body>& bodies_);
		virtual void setBodies(std::vector<Body>&& bodies_);
		virtual void extractLocalBodiesTo(std::vector<Body>& result);

		// iterator classes for traversing of pointered node data structures
		template<typename Derived>
		class BaseNodeIterator {
		protected:
			Node* node_{ nullptr };
			BaseNodeIterator(Node * node) :node_(node) {};
		public:
			typedef std::ptrdiff_t difference_type;
			typedef Node value_type;
			typedef Node& reference;
			typedef Node* pointer;
			typedef std::bidirectional_iterator_tag iterator_category;

			const BaseNodeIterator& operator++() {
				return static_cast<Derived*>(this)->operator++();
			}
			BaseNodeIterator operator++(int) {
				auto result = *this; ++(*this); return result;
			}
			BaseNodeIterator& operator--() { 
				return static_cast<Derived*>(this)->operator--(); 
			}
			BaseNodeIterator operator--(int) {
				auto result = *this; --(*this); return result;
			}

			reference operator*() const { return *node_; }
			pointer operator->() const { return node_; }
			friend bool operator==(const BaseNodeIterator& lhs, const BaseNodeIterator& rhs) { return lhs.node_ == rhs.node_; }
			friend bool operator!=(const BaseNodeIterator& lhs, const BaseNodeIterator& rhs) { return !(lhs == rhs); }
		};
		struct NodeIterator : BaseNodeIterator<NodeIterator>{
			NodeIterator(Node * node) : BaseNodeIterator(node) {};
			const BaseNodeIterator& operator++() { node_ = node_->next; return *this; }
			BaseNodeIterator& operator--() { node_ = node_->prev; return *this; }

		};
		struct SiblingNodeIterator : BaseNodeIterator<SiblingNodeIterator>{
			SiblingNodeIterator(Node * node) : BaseNodeIterator(node) {};
			const BaseNodeIterator& operator++() { node_ = node_->nextSibling; return *this; }
			BaseNodeIterator& operator--() { node_ = node_->prevSibling; return *this; }
		};
	private:
		class SiblingNodesView {
			Tree* tree_;
		public:
			SiblingNodesView(Tree* tree);
			SiblingNodeIterator begin();
			SiblingNodeIterator end();
			SiblingNodeIterator begin() const;
			SiblingNodeIterator end() const;
		};
		SiblingNodesView siblingNodesView;
	};
} // namespace nbody

#endif
