#pragma once

#include "gsl/gsl"
#include <boost/container/static_vector.hpp>
#include <mpi.h>

#include "Configuration.hpp"
#include "MpiEnvironment.hpp"
#include "State.hpp"
#include "Util.hpp"

class Communicator {
	constexpr static std::size_t NoNeighbors{8};

	template <typename T>
	using Vector = boost::container::static_vector<T, NoNeighbors>;

	CommunicationMode _commMode;
	Vector<int> _neighbors;
	Vector<int> _sizes;
	Vector<MPI_Datatype> _sendTypes;
	const Vector<MPI_Datatype>& _recvTypes{_sendTypes};
	Vector<MPI_Aint> _sendDisplacements;
	Vector<MPI_Aint> _recvDisplacements;
	MPI_Comm _commDistGraph{MPI_COMM_NULL};

	MPI_Datatype _haloRowType;
	MPI_Datatype _haloColumnType;
	MPI_Datatype _haloCornerType{MPI_CHAR};

	void ReportInvalidCommunicationModeError() const;

  public:
	Communicator() = default;
	Communicator(const MpiEnvironment& env, CommunicationMode commMode,
	             const Size& gridSize, const Size& tileSize);
	~Communicator();

	void swap(Communicator& first, Communicator& second);

	Communicator(Communicator&) = delete;
	Communicator& operator=(Communicator&) = delete;
	Communicator(Communicator&& other) noexcept;
	Communicator& operator=(Communicator&& other) noexcept;

	void Communicate(gsl::multi_span<State, -1, -1>& model);

	class MpiRequest {
	  public:
		template <typename T>
		using DoubleVector =
		    boost::container::static_vector<T, NoNeighbors * 2>;

	  private:
		DoubleVector<MPI_Request> _reqs;
		bool finished{};

	  public:
		MpiRequest(DoubleVector<MPI_Request> reqs);
		void Wait();
		~MpiRequest();
	};

	MpiRequest AsyncCommunicate(gsl::multi_span<State, -1, -1>& model);
};