# -*- coding: utf-8 -*-
# bin/cs_config.py.  Generated from cs_config.py.in by configure.

#-------------------------------------------------------------------------------

# This file is part of Code_Saturne, a general-purpose CFD tool.
#
# Copyright (C) 1998-2018 EDF S.A.
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
# Street, Fifth Floor, Boston, MA 02110-1301, USA.

#-------------------------------------------------------------------------------

import os
import sys
from optparse import OptionParser

#-------------------------------------------------------------------------------

# Prerequisites libraries
#------------------------

class prerequisite:

    def __init__(self, name, have = "no",
                 variant = None, dynamic_load = False,
                 prefix = None, execprefix = None,
                 bindir = None, includedir = None, libdir = None,
                 flags = None):

        # Library name
        self.name = name

        # Have
        self.have = have

        # Loaded dynamically on demand (linked separately)
        self.dynamic_load = dynamic_load

        # Library variant
        self.variant = variant

        # Library installation directories
        self.prefix = prefix
        self.execprefix = execprefix
        self.bindir = bindir
        self.includedir = includedir
        self.libdir = libdir

        # Library build (dictionnary {cppflags, ldflags, libs})
        self.flags = flags

    def print_config(self):

        print("Prequisite: " + self.name)


# Configuration info
#-------------------

class config:

    def __init__(self):

        # List of optionnal features

        self.optfeatures = ['debug', 'relocatable', 'shared',
                            'gui', 'frontend',
                            'mpi', 'openmp', 'socket',
                            'long-gnum', 'nls', 'host']
        self.features = {}

        # List of mandatory and optionnal libraries to link with
        # The order is important so as to have a coherent link command

        self.deplibs = ['saturne',                      # Code_Saturne
                        'ple',                          # PLE
                        'eos', 'freesteam', 'coolprop', # Equations of state
                        'ccm', 'cgns', 'med', 'hdf5',   # Mesh filters
                        'catalyst', 'melissa',          # co-processing libraries
                        'medcoupling',                  # MED coupling
                        'petsc',                        # Linear algebra
                        'metis', 'scotch',              # Partionning libraries
                        'mpi',                          # MPI
                        'blas',                         # BLAS (benchmark use)
                        'system']                       # User & system libraries

        # Compilers, flags and special commands

        self.compilers = {'cc': "mpicc",
                          'cxx': "mpicxx",
                          'fc': "mpif90",
                          'ld': "mpicc",
                          'version': "@CC_VERSION@"}

        self.flags = {'cflags': "-strict-ansi -std=c99 -restrict -funsigned-char -Wall -Wcheck -Wshadow -Wpointer-arith -Wmissing-prototypes -Wuninitialized -Wunused -wd981 -O3 -xCORE-AVX512 -Ofast -qopenmp  -O2",
                      'cxxflags': "-W -Wall -Wshadow -Wpointer-arith -Wcast-qual -Wcast-align -Wwrite-strings -Wunused -Wfloat-equal -std=gnu++0x  -funroll-loops -O2 -Wuninitialized",
                      'fcflags': "-cpp -fpic -warn -diag-disable 7712 -O3 -xCORE-AVX512 -Ofast -qopenmp  -O2"}

        self.fcmodinclude = "-I"
        self.rpath = ""
        self.special_user_link = ""

        # Constants for system-dependant file extensions
        if sys.platform.startswith("win"):
            self.cfgext = ".ini"
            self.exeext = ".exe"
            self.shext = ".bat"
        else:
            self.cfgext = ".cfg"
            self.exeext = ""
            self.shext = ""

        # Libraries to add for specific executable names

        self.exec_libs = {'cs_solver' + self.exeext: "-lcs_solver",
                          'nc_solver' + self.exeext: "-lneptune"}

        # Python-related information

        self.python = "/usr/bin/python"
        self.pyuic4 = ""
        self.pyrcc4 = ""
        self.pyuic5 = ""
        self.pyrcc5 = ""

        # Execution environment

        self.env_modules = "no"
        self.env_modulecmd = ""

        # SALOME-related information

        self.have_salome = "no"
        self.salome_env = ""
        self.salome_run = ""
        self.salome_kernel = ""

        # Setup the optionnal features

        self.features['debug'] = "no"
        self.features['relocatable'] = "no"
        self.features['shared'] = "no"
        self.features['gui'] = "no"
        self.features['frontend'] = "yes"
        self.features['mpi'] = "yes"
        self.features['openmp'] = "yes"
        self.features['long-gnum'] = "yes"
        self.features['nls'] = 'yes'
        self.features['build_os'] = "linux-gnu"

        # Now, one can setup the prerequisites information

        self.libs = {}

        # Setup Code_Saturne libraries
        # Here, CPPFLAGS and LDFLAGS will be provided by a get_dir method

        self.libs['saturne'] = \
            prerequisite("Code_Saturne",
                         have = "yes",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': "-lsaturne"})

        # Setup PLE library
        # Here, the variant (internal or external) will be used to add
        # paths to the command line

        self.libs['ple'] = \
            prerequisite("PLE",
                         have = "yes",
                         variant = "internal",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': "-lple"})

        # Setup user and system libraries

        self.libs['system'] = \
            prerequisite("System",
                         have = "yes",
                         flags = {'cppflags': "  -DNDEBUG ",
                                  'ldflags': "  -O -qopenmp ",
                                  'libs': " -lz    -L/opt/mpi/openmpi-icc/2.0.4.4.0.xcea/lib -L/ccc/products2/ifort-17.0.6.256/Atos_7__x86_64/system/default/compilers_and_libraries_2017.6.256/linux/compiler/lib/intel64_lin -L/usr/lib/gcc/x86_64-redhat-linux/4.8.5/ -L/usr/lib/gcc/x86_64-redhat-linux/4.8.5/../../../../lib64 -L/usr/lib/gcc/x86_64-redhat-linux/4.8.5/../../../../lib64/ -L/lib/../lib64 -L/lib/../lib64/ -L/usr/lib/../lib64 -L/usr/lib/../lib64/ -L/usr/lib/gcc/x86_64-redhat-linux/4.8.5/../../../ -L/lib64 -L/lib/ -L/usr/lib64 -L/usr/lib -lmpi_usempif08 -lmpi_usempi_ignore_tkr -lmpi_mpifh -lmpi -lifport -lifcoremt_pic -limf -lsvml -lm -lipgo -lirc -lpthread -lirc_s -ldl"})

        # Setup the optionnal libraries

        self.libs['blas'] = \
            prerequisite("BLAS",
                         have = "no",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['ccm'] = \
            prerequisite("CCM",
                         have = "no",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['cgns'] = \
            prerequisite("CGNS",
                         have = "no",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['hdf5'] = \
            prerequisite("HDF5",
                         have = "no",
                         flags = {'cppflags': "  -DOMPI_SKIP_MPICXX",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['med'] = \
            prerequisite("MED",
                         have = "no",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['catalyst'] = \
            prerequisite("CATALYST",
                         have = "no",
                         dynamic_load = False,
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['melissa'] = \
            prerequisite("MELISSA",
                         have = "no",
                         dynamic_load = False,
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['medcoupling'] = \
            prerequisite("MEDCOUPLING",
                         have = "no",
                         dynamic_load = False,
                         flags = {'cppflags': "-I/include",
                                  'ldflags': "-L/lib",
                                  'libs': ""})

        self.libs['eos'] = \
            prerequisite("EOS",
                         have = "no",
                         prefix="no",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['freesteam'] = \
            prerequisite("FREESTEAM",
                         have = "no",
                         prefix="check",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['coolprop'] = \
            prerequisite("COOLPROP",
                         have = "no",
                         prefix="check",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': "",
                                  'pythonpath': ""})

        self.libs['metis'] = \
            prerequisite("METIS",
                         have = "no",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['mpi'] = \
            prerequisite("MPI",
                         have = "yes",
                         variant = "OpenMPI",
                         bindir = "",
                         libdir = "/opt/mpi/openmpi-icc/2.0.4.4.0.xcea/lib",
                         flags = {'cppflags': " -DOMPI_SKIP_MPICXX",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['petsc'] = \
            prerequisite("PETSc",
                         have = "no",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': ""})

        self.libs['scotch'] = \
            prerequisite("SCOTCH",
                         have = "no",
                         flags = {'cppflags': "",
                                  'ldflags': "",
                                  'libs': ""})

    def __get_search_paths_catalyst__(self):
        """
        return Catalyst library search path, Python search paths,
        and other environment variables if available
        """

        lib_dirs = []
        pythonpath_dirs = []
        env_vars = None

        catalyst_lib_dir = None

        libs = self.libs['catalyst'].flags['libs']
        for l in libs.split('-Wl,'):
            if l.find('libvtkPVPythonCatalyst') > -1:
                catalyst_lib_dir = os.path.dirname(l)
                break

        if catalyst_lib_dir and self.features['relocatable']:
            catalyst_root_dir = os.getenv('CATALYST_ROOT_DIR')
            if catalyst_root_dir:
                subdir_idx = catalyst_lib_dir.rfind('lib')
                if subdir_idx > 1:
                    catalyst_lib_dir = os.path.join(catalyst_root_dir,
                                                    catalyst_lib_dir[subdir_idx:])

        if catalyst_lib_dir:
            if self.libs['catalyst'].dynamic_load:
                lib_dirs = [catalyst_lib_dir]
            sp_dir = os.path.join(catalyst_lib_dir, 'site-packages')
            if os.path.isdir(sp_dir):
                pythonpath_dirs = [sp_dir]

        # Add possible additional Catalyst dependency paths

        catalyst_ld_add_path = os.getenv('CATALYST_LD_ADD_PATH')
        if catalyst_ld_add_path:
            for d in catalyst_ld_add_path.split(os.pathsep):
                if d: # avoid empty values before first or after last separator
                    lib_dirs.append(d)

        # Add additional environment variables

        catalyst_plugin_path = os.getenv('CATALYST_PLUGIN_PATH')
        if not catalyst_plugin_path:
            catalyst_plugin_path = ''
        env_vars = {'PV_PLUGIN_PATH':catalyst_plugin_path}

        return lib_dirs, pythonpath_dirs, env_vars

    def get_run_environment_dependencies(self):
        """
        return library search path, Python search paths,
        and other environment variables if available or required
        """

        lib_dirs = []
        pythonpath_dirs = []
        env_vars = {}

        for lib in self.deplibs:
            if self.libs[lib].have == "yes":

                if lib == 'catalyst':
                    catalyst_lib_dirs, catalyst_pythonpath_dirs, catalyst_env_vars \
                        = self.__get_search_paths_catalyst__()

                    for d in catalyst_lib_dirs:
                        lib_dirs.append(d)
                    for p in catalyst_pythonpath_dirs:
                        pythonpath_dirs.append(p)
                    env_vars.update(catalyst_env_vars)

        return lib_dirs, pythonpath_dirs, env_vars

    def __get_dep_libs_path_catalyst__(self):
        """
        return Catalyst dependency path required for compilation
        """

        lib_dirs = []

        # Add possible additional Catalyst dependency paths

        catalyst_ld_add_path = os.getenv('CATALYST_LD_ADD_PATH')
        if catalyst_ld_add_path:
            for d in catalyst_ld_add_path.split(os.pathsep):
                if d: # avoid empty values before first or after last separator
                    lib_dirs.append(d)

        return lib_dirs

    def get_compile_dependency_paths(self):
        """
        return additional library search if available or required
        """

        lib_dirs = []

        for lib in self.deplibs:
            if self.libs[lib].have == "yes":

                if lib == 'catalyst':
                    catalyst_lib_dirs = self.__get_dep_libs_path_catalyst__()
                    for d in catalyst_lib_dirs:
                        lib_dirs.append(d)

        return lib_dirs

    def print_config(self):
        """
        Print configuration info
        """

        for lib in self.deplibs:
            self.libs[lib].print_config()

#-------------------------------------------------------------------------------

def process_cmd_line(argv):
    """
    Processes the passed command line arguments.

    Input Argument:
      arg -- This can be either a list of arguments as in
             sys.argv[1:] or a string that is similar to the one
             passed on the command line.  If it is a string,
             it is split to create a list of arguments.
    """

    parser = OptionParser(usage="usage: %prog [options]")

    parser.add_option("--cc", dest="print_cc",
                      action="store_true",
                      help="C compiler used for build")

    parser.add_option("--cxx", dest="print_cxx",
                      action="store_true",
                      help="C++ compiler used for build")

    parser.add_option("--fc", dest="print_fc",
                      action="store_true",
                      help="Fortran compiler used for build")

    parser.add_option("--cflags", dest="print_cflags",
                      action="store_true",
                      help="C compiler flags")

    parser.add_option("--cxxflags", dest="print_cxxflags",
                      action="store_true",
                      help="C++ compiler flags")

    parser.add_option("--fcflags", dest="print_fcflags",
                      action="store_true",
                      help="Fortran compiler flags")

    parser.add_option("--rpath", dest="print_rpath",
                      action="store_true",
                      help="Linker rpath command line")

    parser.add_option("--python", dest="print_python",
                      action="store_true",
                      help="Python interpreter")

    parser.add_option("--pyuic4", dest="print_pyuic4",
                      action="store_true",
                      help="pyuic4 tool for PyQt4 support")

    parser.add_option("--pyrcc4", dest="print_pyrcc4",
                      action="store_true",
                      help="pyrcc4 tool for PyQt4 support")

    parser.add_option("--pyuic5", dest="print_pyuic5",
                      action="store_true",
                      help="pyuic5 tool for PyQt5 support")

    parser.add_option("--pyrcc5", dest="print_pyrcc5",
                      action="store_true",
                      help="pyrcc5 tool for PyQt5 support")

    parser.add_option("--have", dest="have", metavar="<lib>",
                      help="supported feature or library")

    parser.add_option("--cppflags", dest="cppflags", metavar="<lib>",
                      help="C preprocessor flags (e.g. -D<macro>, ...)")

    parser.add_option("--ldflags", dest="ldflags", metavar="<lib>",
                      help="linker flags (e.g. -g, -L<path>, ...)")

    parser.add_option("--libs", dest="libs", metavar="<lib>",
                      help="librairies used (e.g. -l<libname>, ...)")

    parser.add_option("--pythondir", dest="print_pythondir",
                      action="store_true",
                      help="directory for the 'site-packages' subdirectory" \
                          " of the standard Python install tree")

    parser.add_option("--datarootdir", dest="print_datarootdir",
                      action="store_true",
                      help="directory where architecture-independent" \
                          + " files are installed (e.g. <prefix>/share)")

    parser.set_defaults(print_cc=False)
    parser.set_defaults(print_cxx=False)
    parser.set_defaults(print_fc=False)

    parser.set_defaults(print_cflags=False)
    parser.set_defaults(print_cxxflags=False)
    parser.set_defaults(print_fcflags=False)

    parser.set_defaults(print_rpath=False)

    parser.set_defaults(print_python=False)
    parser.set_defaults(print_pyrcc4=False)
    parser.set_defaults(print_pyuic4=False)
    parser.set_defaults(print_pyrcc5=False)
    parser.set_defaults(print_pyuic5=False)

    parser.set_defaults(have=None)
    parser.set_defaults(cppflags=None)
    parser.set_defaults(ldflags=None)
    parser.set_defaults(libs=None)

    parser.set_defaults(print_pythondir=False)
    parser.set_defaults(print_datarootdir=False)

    (options, args) = parser.parse_args(argv)

    if len(args) > 0:
        parser.print_help()
        sys.exit(1)

    return options

#-------------------------------------------------------------------------------

def get_config(pkg):
    """
    Get the configuration information.
    """
    msg = """\
Compilers and associated options:
  cc = %(cc)s
  cxx = %(cxx)s
  fc = %(fc)s
  cflags = %(cflags)s
  cxxflags = %(cxxflags)s
  fcflags = %(fcflags)s
  rpath = %(rpath)s\
"""

    return msg \
        % { 'cc':pkg.config.compilers['cc'],
            'cxx': pkg.config.compilers['cxx'],
            'fc':pkg.config.compilers['fc'],
            'cflags':pkg.config.flags['cflags'],
            'cxxflags':pkg.config.flags['cxxflags'],
            'fcflags':pkg.config.flags['fcflags'],
            'rpath':pkg.config.rpath }

#-------------------------------------------------------------------------------
# Main
#-------------------------------------------------------------------------------

def main(argv, pkg):
    """
    Main configuration info function.
    """

    opts = process_cmd_line(argv)

    cfg = pkg.config

    if opts.print_cc  == True: print(cfg.compilers['cc'])
    if opts.print_cxx == True: print(cfg.compilers['cxx'])
    if opts.print_fc  == True: print(cfg.compilers['fc'])

    if opts.print_cflags   == True: print(cfg.flags['cflags'])
    if opts.print_cxxflags == True: print(cfg.flags['cxxflags'])
    if opts.print_fcflags  == True: print(cfg.flags['fcflags'])

    if opts.print_rpath == True: print(cfg.rpath)

    if opts.print_python  == True: print(cfg.python)
    if opts.print_pyuic4  == True: print(cfg.pyuic4)
    if opts.print_pyrcc4  == True: print(cfg.pyrcc4)
    if opts.print_pyuic5  == True: print(cfg.pyuic5)
    if opts.print_pyrcc5  == True: print(cfg.pyrcc5)

    if opts.have is not None:
        if opts.have in cfg.deplibs: print(cfg.libs[opts.have].have)
        if opts.have in cfg.optfeatures: print(cfg.features[opts.have])

    if opts.cppflags is not None:
        # Specific handling of Code_Saturne has pkgincludedir has to be
        # correctly expended. Likewise for PLE, if internal version is used
        if opts.cppflags == "saturne":
            print("-I" + pkg.get_dir("pkgincludedir"))
        elif opts.cppflags == "ple":
            if cfg.libs['ple'].variant == "internal":
                print("-I" + pkg.get_dir("includedir"))
            else:
                print(cfg.libs[opts.cppflags].flags['cppflags'])
        else:
            print(cfg.libs[opts.cppflags].flags['cppflags'])

    if opts.ldflags is not None:
        # Specific handling of Code_Saturne has pkgincludedir has to be
        # correctly expended. Likewise for PLE, if internal version is used
        if opts.ldflags == "saturne":
            print("-L" + pkg.get_dir("libdir"))
        elif opts.ldflags == "ple":
            if cfg.libs['ple'].variant == "internal":
                print("-L" + pkg.get_dir("libdir"))
            else:
                print(cfg.libs[opts.cppflags].flags['ldflags'])
        else:
            if cfg.libs[opts.ldflags].dynamic_load == False:
                print(cfg.libs[opts.ldflags].flags['ldflags'])

    if opts.libs is not None:
        if cfg.libs[opts.libs].dynamic_load == False:
            print(cfg.libs[opts.libs].flags['libs'])

    if opts.print_pythondir: print(pkg.get_dir("pythondir"))
    if opts.print_datarootdir: print(pkg.get_dir("datarootdir"))

if __name__ == '__main__':
    import sys
    import cs_package
    pkg = cs_package.package()
    main(sys.argv[1:], pkg)

